/* --COPYRIGHT--,BSD
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//******************************************************************************
//  MSP430G2452 Multiple freq. PWM Demo - Timer_A, Toggle P1.0-2, P1.4
//
//  Description: Use Timer_A CCRx units and overflow to generate three
//  independent PWM freqs and duty cycles. For demonstration, CCR0 and CCR1 output
//  units are optionally selected with port pins P1.1, P1.2, and P1.4 in toggle
//  mode. Interrupts are also enabled with all CCRx units,
//  software loads offset to next interval only - as long as the interval offset
//  is aded to CCRx, toggle rate is generated in hardware. Timer_A overflow ISR
//  is used to toggle P1.0 with software. 
//
//  ACLK = n/a, MCLK = SMCLK = TACLK = default DCO ~1MHz
//  As coded and assuming ~1MHz DCO, toggle rates are:
//  P1.1 = CCR0 ~ 1MHz/(50 + 150) = ~5kHz @ 25% duty cycle
//  P1.2 = CCR1 ~ 1MHz/(50 + 350) = ~2.5kHz @ 12.5% duty cycle
//	P1.4 = CCR2 ~ 1MHz/(600 + 400) = ~1kHz @ 60% duty cycle
//
//  P1.0 = overflow ~ 1MHz/(2*65536) ~8Hz
//
//               MSP430G2452
//            -----------------
//        /|\|              XIN|-
//         | |                 |
//         --|RST          XOUT|-
//           |                 |
//           |         P1.1/TA0|--> TACCR0
//           |         P1.2/TA1|--> TACCR1
//           |         P1.4/TA2|--> TACCR2
//           |             P1.0|--> Overflow/software
//
//******************************************************************************

#include  <msp430.h>

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  
  //Calibrate DCO for 1MHz operation
  BCSCTL1 = CALBC1_1MHZ;
  DCOCTL = CALDCO_1MHZ;

  P1SEL |= 0x16;                            // P1.1 - P1.2, P1.4 option select
  P1SEL2 |= 0x10;                           // P1.4 option select
  P1DIR |= 0x17;                            // P1.0 - P1.2, P1.4 outputs
  
  TACCTL0 = CCIS_0 + OUTMOD_4 + CCIE;       // CCR0 toggle, interrupt enabled, CCI0A input on CCI (check device datasheet for pin connection)
  TACCTL1 = CCIS_0 + OUTMOD_4 + CCIE;       // CCR1 toggle, interrupt enabled, CCI1A input on CCI (check device datasheet for pin connection)
  TACCTL2 = CCIS_0 + OUTMOD_4 + CCIE;       // CCR2 toggle, interrupt enabled, CCI2A input on CCI (check device datasheet for pin connection)
  TACTL = TASSEL_2 +  MC_2 + TAIE;          // SMCLK, Contmode, int enabled

  _BIS_SR(LPM0_bits + GIE);                 // Enter LPM0 w/ interrupt
}

// Timer A0 interrupt service routine
#pragma vector=TIMER0_A0_VECTOR
__interrupt void Timer_A0 (void)
{
    if(TACCTL0 & CCI)                         // If output currently high
    {
        TACCR0 += 50;                         // 25% high
    }
    else
    {
        TACCR0 += 150;                        // 75% low
    }
}

// Timer A1 Interrupt Vector (TA0IV) handler
#pragma vector=TIMER0_A1_VECTOR
__interrupt void Timer_A1(void)
{
  switch( TA0IV )
  {
  case  2:  if(TACCTL1 & CCI)                 // If output currently high
            {
                TACCR1 += 50;                 // 12.5% high
            }
            else
            {
                TACCR1 += 350;                // 87.5% low
            }
           break;
  case  4:  if(TACCTL2 & CCI)                 // If output currently high
            {
                TACCR2 += 600;                // 60% high
            }
            else
            {
                TACCR2 += 400;                // 40% low
            }
           break;
  case 10: P1OUT ^= 0x01;                   // Timer overflow
           break;
  default: break;
  }
}

